package tests;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;

import association.Evenement;
import association.GestionEvenements;
import association.InformationPersonnelle;
import association.Membre;
import java.time.Month;
import java.util.List;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

/**
 * Tests JUnit de la classe Membre {@link association.Membre}.
 *
 * @author Maxime André
 * @see association.Membre
 */
class TestMembre {
  
  /**
   * Un premier membre.
   */
  private Membre membre;
  
  /**
   * Une deuxième membre.
   */
  private Membre membre2;
  
  /**
   * Un premier évènement.
   */
  private Evenement event1;
  
  /**
   * Un second évènement.
   */
  private Evenement event2;
  
  /**
   * Un troisuème évènement.
   */
  private Evenement event3;
  
  /**
   * Une information personnelle complète.
   */
  private InformationPersonnelle infoperso;
  
  /**
   * Une information personnelle avec les champs obligatoire uniquement.
   */
  private InformationPersonnelle infObligatoire;
  
  /**
   * Instancie deux membres, une information personnelle complète, une
   * information personnelle avec les champs obligatoire et trois évènements
   * pour les tests. Définie les informations personnelles des deux membres.
   */
  @BeforeEach
  void setUp() throws Exception {
    membre = new Membre();
    membre2 = new Membre();
    infoperso = new InformationPersonnelle("Gambrelle", "Eric", "Lyon", 23);
    infObligatoire = new InformationPersonnelle("Xavier", "Charle", "", 0);
    membre.definirInformationPersonnelle(infoperso);
    membre2.definirInformationPersonnelle(infObligatoire);
    event1 = new Evenement("Tournois d'echecs", "Pl. de Strasbourg", 23,
        Month.DECEMBER, 2022, 13, 20, 4, 16);
    event2 = new Evenement("Karting", "Av. 1ère Dfl", 5, Month.OCTOBER, 2022,
        14, 0, 2, 6);
    event3 = new Evenement("Tombola", "Rue de Touraine", 6, Month.JANUARY, 2023,
        14, 50, 1, 25);
  }
  
  /**
   * Ne fait rien aprés les tests : à modifier au besoin.
   *
   * @throws Exception ne peut pas être levée ici
   */
  @AfterEach
  void tearDown() throws Exception {}
  
  /**
   * Vérifie que la liste d'évènements du membre ainsi que la liste d'évènements
   * du membre à venir sont bien initialisé à null lors de sa création.
   */
  @Test
  void testInitMembre() {
    Membre mbr = new Membre();
    assertTrue(mbr.getInfoperso() == null);
    assertTrue(mbr.ensembleEvenements().isEmpty() == true);
  }
  
  /**
   * Vérifie que les informations du membre sont bien définis.
   */
  @Test
  void testDefInfoMembre() {
    assertTrue(membre.getInformationPersonnelle() != null);
  }
  
  /**
   * Vérifie qu'on peut positionner l'âge d'un membre à 62 ans.
   */
  @Test
  void testSetAge62() {
    membre2.getInfoperso().setAge(62);
    assertEquals(membre2.getInfoperso().getAge(), 62);
  }
  
  /**
   * Vérifie qu'on peut pas positionner l'âge d'un membre en âge négatif.
   */
  @Test
  void testSetAgeNegatif() {
    membre2.getInfoperso().setAge(-34);
    assertTrue(membre2.getInfoperso().getAge() != -34);
  }
  
  /**
   * Vérifie qu'on ne peut pas positionner l'adresse d'un membre à null.
   */
  @Test
  void testSetAdresseNull() {
    membre2.getInfoperso().setAdresse(null);
    assertTrue(membre2.getInfoperso().getAdresse() != null);
  }
  
  /**
   * Vérifie qu'on peut modifier l'âge d'un membre à 55 ans.
   */
  @Test
  void testModifAge22() {
    membre.getInfoperso().setAge(55);
    assertEquals(membre.getInfoperso().getAge(), 55);
  }
  
  /**
   * Vérifie qu'on ne peut pas modifier l'âge d'un membre dont les informations
   * personnelles son complètes en âge négatif.
   */
  @Test
  void testAgeNegatif() {
    membre.getInfoperso().setAge(-23);
    assertTrue(membre.getInfoperso().getAge() != -23);
  }
  
  /**
   * Vérifie qu'on ne peut pas modifier l'adresse d'un membre dont les
   * informations personnelles sont complètes à null.
   */
  @Test
  void testAdresseNull() {
    membre.getInfoperso().setAdresse(null);
    assertTrue(membre.getInfoperso().getAdresse() != null);
  }
  
  /**
   * Vérifie que les informations personnelles sont bien associées au membre.
   */
  @Test
  void testAssociationInfoMembre() {
    Membre newMembre = new Membre();
    InformationPersonnelle info =
        new InformationPersonnelle("Allan", "Teixeira", "Brest", 21);
    newMembre.definirInformationPersonnelle(info);
    assertEquals(newMembre.getInfoperso().getNom(), "Allan");
    assertEquals(newMembre.getInfoperso().getPrenom(), "Teixeira");
    assertTrue(newMembre.getInfoperso().getAdresse() != null);
    assertEquals(newMembre.getInfoperso().getAge(), 21);
  }
  
  /**
   * Vérifie que les information personnelles obligatoires sont bien associées
   * au membre.
   */
  @Test
  void testAssocationInfoObligatoire() {
    Membre newMbr = new Membre();
    InformationPersonnelle inf = new InformationPersonnelle("Bob", "Leroi");
    newMbr.definirInformationPersonnelle(inf);
    assertEquals(newMbr.getInfoperso().getNom(), "Bob");
    assertEquals(newMbr.getInfoperso().getPrenom(), "Leroi");
    assertEquals(newMbr.getInfoperso().getAdresse(), "");
    assertTrue(newMbr.getInfoperso().getAge() == 0);
  }
  
  /**
   * Vérifie que la liste des évènements du membre est bien renvoyé. Dans les
   * cas où, la liste est vide, le membre est inscrit à des évènements,
   * l'évènement et quand le membre est désinscrit des évènements.
   */
  @Test
  void testEventMembre() {
    // Test quand la liste est encore vide
    assertTrue(membre.ensembleEvenements().isEmpty() == true);
    assertTrue(membre.ensembleEvenementsAvenir().isEmpty() == true);
    GestionEvenements gestEvent =
        new GestionEvenements(membre.ensembleEvenements());
    
    // Inscription du membre aux évènements
    assertTrue((gestEvent.inscriptionEvenement(event1, membre)) == true);
    assertTrue((gestEvent.inscriptionEvenement(event2, membre)) == true);
    assertTrue((gestEvent.inscriptionEvenement(event3, membre)) == true);
    
    // Test quand la liste n'est pas vide
    assertTrue(membre.ensembleEvenements().isEmpty() == false);
    assertTrue(membre.ensembleEvenementsAvenir().isEmpty() == false);
    
    List<Evenement> list1 = membre.ensembleEvenements();
    List<Evenement> list2 = membre.ensembleEvenementsAvenir();
    
    // Test pour la liste des évènements du membre
    assertTrue(list1.contains(event1) == true || list1.contains(event2) == true
        || list1.contains(event3) == true);
    
    // Test pour la liste des évènements du membre à venir
    assertTrue(list2.contains(event1) == true || list2.contains(event2) == false
        || list2.contains(event3) == true);
    
    // suppression d'un évènement
    gestEvent.supprimerEvenement(event3);
    
    // Mise à jour des listes d'évènements du membre
    list1 = membre.ensembleEvenements();
    list2 = membre.ensembleEvenementsAvenir();
    
    // Vérifie que le membre est désinscrit de l'évènement supprimé
    assertTrue(list1.contains(event3) == false);
    assertTrue(list2.contains(event3) == false);
    
    // Le membre est désinscrit des évènements annulés
    assertTrue((gestEvent.annulerEvenement(event1, membre)) == true);
    assertTrue((gestEvent.annulerEvenement(event2, membre)) == true);
    
    // Test quand le membre n'est plus inscrit aux évènements
    assertTrue(membre.ensembleEvenements().isEmpty() == true);
    assertTrue(membre.ensembleEvenementsAvenir().isEmpty() == true);
  }
}
