package tests;

import static org.junit.jupiter.api.Assertions.assertTrue;

import association.Evenement;
import association.GestionEvenements;
import association.InterGestionEvenements;
import association.InterMembre;
import association.Membre;
import java.time.LocalDateTime;
import java.time.Month;
import java.util.ArrayList;
import java.util.List;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

/**
 * Tests JUnit de la classe {@link association.GestionEvenements}.
 *
 * @author Jarod Martin
 * @see association.GestionEvenements
 */
public class TestGestionEvenements {
  
  /**
   * Instance de GestionEvenements.
   */
  private InterGestionEvenements gestionEvenements;
  
  /**
   * Instancie un un objet GestionEvenements.
   *
   * @throws Exception ne peut pas être levée ici
   */
  @BeforeEach
  void setUp() throws Exception {
    this.gestionEvenements = new GestionEvenements();
  }
  
  /**
   * Rien à effectuer après les tests.
   *
   * @throws Exception ne peut pas être levée ici
   */
  @AfterEach
  void tearDown() throws Exception {}
  
  /**
   * Vérifie que la création d'événements fonctionne bien.
   */
  @Test
  void testCreeEvenement() {
    Evenement event = this.gestionEvenements.creerEvenement("Test", "Lieu", 1,
        Month.APRIL, 2023, 23, 5, 120, 4);
    /* Test événement bien ajouté. */
    assertTrue(this.gestionEvenements.ensembleEvenements().contains(event));
    /* Test création événement incorrect (date). */
    event = this.gestionEvenements.creerEvenement("Test", "Lieu2", 40,
        Month.APRIL, 2023, 23, 5, 120, 4);
    assertTrue(event == null);
    event = this.gestionEvenements.creerEvenement("Test", "Lieu2", 1, null,
        2023, 23, 5, 120, 4);
    assertTrue(event == null);
    event = this.gestionEvenements.creerEvenement("Test", "Lieu2", 1,
        Month.APRIL, 2023, 25, 5, 120, 4);
    assertTrue(event == null);
    event = this.gestionEvenements.creerEvenement("Test", "Lieu2", 1,
        Month.APRIL, 2023, 23, -6, 120, 4);
    assertTrue(event == null);
    /* Test création événement en conflit. */
    event = this.gestionEvenements.creerEvenement("Test", "Lieu", 1,
        Month.APRIL, 2023, 23, 30, 30, 4);
    assertTrue(event == null);
  }
  
  @Test
  void testEnsembleEvenementAvenir() {
    Evenement event1 = this.gestionEvenements.creerEvenement("Nom", "AutreLieu",
        1, Month.APRIL, 2023, 23, 30, 30, 4);
    Evenement event2 = this.gestionEvenements.creerEvenement("Nom",
        "AutreLieu2", 5, Month.APRIL, 2023, 23, 30, 30, 4);
    event1.setDate(LocalDateTime.now().minusDays(30));
    event2.setDate(LocalDateTime.now().plusDays(30));
    /* event1 n'est pas présent. */
    assertTrue(
        !this.gestionEvenements.ensembleEvenementAvenir().contains(event1));
    /* event2 est présent. */
    assertTrue(
        this.gestionEvenements.ensembleEvenementAvenir().contains(event2));
  }
  
  /**
   * Vérifier que l'inscription d'un participant fonctionne.
   */
  @Test
  void testInscriptionParticipant() {
    final Evenement event1 = this.gestionEvenements.creerEvenement("Event1",
        "Lieu1", 2, Month.JANUARY, 2023, 15, 0, 120, 1);
    final Evenement event3 = this.gestionEvenements.creerEvenement("Event3",
        "Lieu3", 12, Month.APRIL, 2023, 12, 12, 60, 5);
    final InterMembre membre1 = new Membre();
    final InterMembre membre2 = new Membre();
    /* Test d'ajout d'un membre à un événement. */
    assertTrue(this.gestionEvenements.inscriptionEvenement(event1, membre1));
    assertTrue(event1.getParticipants().contains(membre1));
    /* Test d'ajout d'un événement sans conflit. */
    assertTrue(this.gestionEvenements.inscriptionEvenement(event3, membre1));
    assertTrue(event3.getParticipants().contains(membre1));
    /*
     * Test d'ajout d'un membre alors que le nombre de membre maximum est
     * atteint.
     */
    assertTrue(!this.gestionEvenements.inscriptionEvenement(event1, membre2));
    assertTrue(!event1.getParticipants().contains(membre2));
    /* Test d'ajout d'un membre dans un événement où il y a déjà un membre */
    assertTrue(this.gestionEvenements.inscriptionEvenement(event3, membre2));
    assertTrue(event3.getParticipants().contains(membre2));
  }
  
  /**
   * Vérifier que la désinscription d'un participant fonctionne.
   */
  @Test
  void testSuppressionParticipant() {
    Evenement event = this.gestionEvenements.creerEvenement("Event1", "Lieu1",
        2, Month.JANUARY, 2023, 15, 0, 120, 1);
    InterMembre membre = new Membre();
    this.gestionEvenements.inscriptionEvenement(event, membre);
    /* Suppression d'un membre présent. */
    assertTrue(this.gestionEvenements.annulerEvenement(event, membre));
    assertTrue(!event.getParticipants().contains(membre));
    /* Suppression d'un membre non présent. */
    assertTrue(!this.gestionEvenements.annulerEvenement(event, membre));
  }
  
  /**
   * Vérifie l'initialisation avec une liste d'événements.
   */
  @Test
  void testConstructeur() {
    List<Evenement> evenements = new ArrayList<>();
    GestionEvenements gestion = new GestionEvenements(evenements);
    assertTrue(gestion.ensembleEvenements() != null);
    assertTrue(gestion.ensembleEvenements() == evenements);
  }
  
}
