package tests;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;

import association.Evenement;
import association.InterMembre;
import association.Membre;
import java.time.LocalDateTime;
import java.time.Month;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

/**
 * Tests JUnit de la classe {@link association.Evenement}.
 *
 * @author Jarod Martin
 * @see association.Evenement
 */
class TestEvenement {
  
  /**
   * Instance d'un événement.
   */
  private Evenement event;
  
  /**
   * Instancie un evenement à partir d'une date LocalDateTime et des
   * informations essentielles associées.
   *
   * @throws Exception ne peut pas être levée ici
   */
  @BeforeEach
  void setUp() throws Exception {
    this.event = new Evenement("Goûter", "Liberté", LocalDateTime.now(), 50, 5);
  }
  
  /**
   * Rien à effectuer après les tests.
   *
   * @throws Exception ne peut pas être levée ici
   */
  @AfterEach
  void tearDown() throws Exception {}
  
  /**
   * Vérifier que l'ont peut modifier le nom d'un événement.
   */
  @Test
  void testModifNom() {
    /* Modification du nom. */
    this.event.setNom("Jeux");
    assertEquals(this.event.getNom(), "Jeux");
    /* Dans le cas où on essaye de mettre un nom null. */
    this.event.setNom(null);
    assertTrue(this.event.getNom() != null);
  }
  
  /**
   * Vérifier que l'ont peut modifier le lieu d'un événement.
   */
  @Test
  void testModifLieu() {
    /* Modification du lieu. */
    this.event.setLieu("Parc Penfeld");
    assertEquals(this.event.getLieu(), "Parc Penfeld");
    /* Dans le cas où on essaye de mettre un lieu null. */
    this.event.setLieu(null);
    assertTrue(this.event.getLieu() != null);
  }
  
  /**
   * Vérifier que l'ont peut modifier la date d'un événement.
   */
  @Test
  void testModifDate() {
    LocalDateTime date = LocalDateTime.now();
    /* Modification de la date. */
    this.event.setDate(date);
    assertEquals(this.event.getDate(), date);
    /* Dans le cas où on essaye de mettre une date null. */
    this.event.setDate(null);
    assertTrue(this.event.getDate() != null);
  }
  
  /**
   * Vérifier que l'ont ne peut pas mettre une durée négative.
   */
  @Test
  void testModifDureeNegative() {
    int duree = this.event.getDuree();
    this.event.setDuree(-20);
    assertEquals(this.event.getDuree(), duree);
  }
  
  /**
   * Vérifier que l'ont peut modifier la durée d'un événement.
   */
  @Test
  void testModifDuree() {
    this.event.setDuree(100);
    assertEquals(this.event.getDuree(), 100);
  }
  
  /**
   * Vérifier que l'ont peut modifier le nombre de participants maximum.
   */
  @Test
  void testModifNbParticipantsMax() {
    this.event.setNbParticipantsMax(10);
    assertEquals(this.event.getNbParticipantsMax(), 10);
  }
  
  /**
   * Vérifier que l'ont ne peut pas mettre un nombre de participants négatif.
   */
  @Test
  void testModifNbParticipantsMaxNegatif() {
    int nbParticipantsMax = this.event.getNbParticipantsMax();
    this.event.setNbParticipantsMax(-2);
    assertEquals(this.event.getNbParticipantsMax(), nbParticipantsMax);
  }
  
  /**
   * Vérifier le chevauchement de deux événements par rapport au lieu.
   */
  @Test
  void testChevauchementLieu() {
    Evenement otherEvent =
        new Evenement("Test", "TestPareil", LocalDateTime.now(), 0, 0);
    /* De base ils ne se chevauchent pas. */
    this.event.setLieu("PasPareil");
    assertTrue(this.event.pasDeChevauchementLieu(otherEvent));
    /* Ils se chevauchent */
    this.event.setLieu("Testlieu");
    otherEvent.setLieu("Testlieu");
    assertTrue(!this.event.pasDeChevauchementLieu(otherEvent));
  }
  
  /**
   * Vérifier le chevauchement de deux événements par rapport au temps.
   */
  @Test
  void testChevauchementTemps() {
    Evenement event1 =
        new Evenement("Event1", "Lieu1", LocalDateTime.now(), 0, 0);
    Evenement event2 =
        new Evenement("Event2", "Lieu2", LocalDateTime.now(), 0, 0);
    event2.setDate(3, Month.JANUARY, 2023, 12, 0);
    event2.setDuree(120);
    /*
     * |-----------| 
     *   |---|
     */
    event1.setDate(3, Month.JANUARY, 2023, 11, 0);
    event1.setDuree(30);
    assertTrue(event1.pasDeChevauchementTemps(event2));
    /*
     * |-----------| |----------|
     */
    event1.setDuree(90);
    assertTrue(!event1.pasDeChevauchementTemps(event2));
    /*
     * |-----------| |----|
     */
    event1.setDate(3, Month.JANUARY, 2023, 12, 0);
    assertTrue(!event1.pasDeChevauchementTemps(event2));
    /*
     * |-----------| |----|
     */
    event1.setDate(3, Month.JANUARY, 2023, 12, 30);
    event1.setDuree(30);
    assertTrue(!event1.pasDeChevauchementTemps(event2));
    /*
     * |-----------| |--------|
     */
    event1.setDuree(90);
    assertTrue(!event1.pasDeChevauchementTemps(event2));
    /*
     * |-----------| |------------|
     */
    event1.setDuree(120);
    assertTrue(!event1.pasDeChevauchementTemps(event2));
    /*
     * |-----------| |----|
     */
    event1.setDate(3, Month.JANUARY, 2023, 15, 0);
    assertTrue(event1.pasDeChevauchementTemps(event2));
    /*
     * |-----------| |----------------|
     */
    event1.setDate(3, Month.JANUARY, 2023, 11, 30);
    event1.setDuree(240);
    assertTrue(!event1.pasDeChevauchementTemps(event2));
    /*
     * |-----------| |----|
     */
    event1.setDate(3, Month.JANUARY, 2023, 14, 0);
    event1.setDuree(60);
    assertTrue(event1.pasDeChevauchementLieu(event2));
  }
  
  /**
   * Vérifier que l'ajout de participants est correct.
   */
  @Test
  void testAjoutParticipant() {
    final Evenement event1 =
        new Evenement("Event1", "Lieu1", 2, Month.JANUARY, 2023, 15, 0, 120, 1);
    final Evenement event2 =
        new Evenement("Event2", "Lieu2", 2, Month.JANUARY, 2023, 16, 0, 60, 10);
    final Evenement event3 =
        new Evenement("Event3", "Lieu3", 12, Month.APRIL, 2023, 12, 12, 60, 5);
    final InterMembre membre1 = new Membre();
    final InterMembre membre2 = new Membre();
    /* Test d'ajout d'un membre à un événement. */
    assertTrue(event1.ajouterParticipant(membre1));
    assertTrue(event1.getParticipants().contains(membre1));
    /* Test d'ajout d'un événement en conflit. */
    assertTrue(!event2.ajouterParticipant(membre1));
    assertTrue(!event2.getParticipants().contains(membre1));
    /* Test d'ajout d'un événement sans conflit. */
    assertTrue(event3.ajouterParticipant(membre1));
    assertTrue(event3.getParticipants().contains(membre1));
    /*
     * Test d'ajout d'un membre alors que le nombre de membre maximum est
     * atteint.
     */
    assertTrue(!event1.ajouterParticipant(membre2));
    assertTrue(!event1.getParticipants().contains(membre2));
    /* Test d'ajout d'un membre dans un événement où il y a déjà un membre */
    assertTrue(event3.ajouterParticipant(membre2));
    assertTrue(event3.getParticipants().contains(membre2));
  }
  
  /**
   * Vérifier que la suppression de participant est correct.
   */
  @Test
  void testSuppressionParticipant() {
    InterMembre membre = new Membre();
    this.event.ajouterParticipant(membre);
    /* Suppression d'un membre présent. */
    assertTrue(this.event.supprimerParticipant(membre));
    assertTrue(!this.event.getParticipants().contains(membre));
    /* Suppression d'un membre non présent. */
    assertTrue(!this.event.supprimerParticipant(membre));
  }
  
  /**
   * Vérifier que le constructeur initialise bien les informations.
   */
  @Test
  void testConstructeur() {
    this.event = new Evenement("Randonnée", "Mont Blanc", 8, Month.JANUARY,
        2023, 8, 0, 280, 18);
    LocalDateTime date = this.event.getDate();
    assertEquals(this.event.getNom(), "Randonnée");
    assertEquals(this.event.getLieu(), "Mont Blanc");
    assertEquals(date.getDayOfMonth(), 8);
    assertEquals(date.getMonth(), Month.JANUARY);
    assertEquals(date.getYear(), 2023);
    assertEquals(date.getHour(), 8);
    assertEquals(date.getMinute(), 0);
    assertEquals(this.event.getDuree(), 280);
    assertEquals(this.event.getNbParticipantsMax(), 18);
  }
  
  /**
   * Vérifier que le constructeur réagit bien à des données invalides passées.
   */
  void testConstructeurParamInvalide() {
    this.event = new Evenement(null, null, null, -10, -20);
    assertTrue(this.event.getNom() != null);
    assertTrue(this.event.getLieu() != null);
    assertTrue(this.event.getDate() != null);
    assertTrue(this.event.getDuree() > 0);
    assertTrue(this.event.getNbParticipantsMax() > 0);
  }
  
  /**
   * Vérification du fonctionnement de la méthode equals redéfinie.
   */
  @Test
  void testEquals() {
    Evenement otherEvent = new Evenement("Cinéma", "Multiplexe Liberté", 2,
        Month.APRIL, 2023, 17, 0, 120, 24);
    /* Événements différents */
    this.event = new Evenement("Randonnée", "Mont Blanc", 8, Month.JANUARY,
        2023, 8, 0, 280, 18);
    assertTrue(!this.event.equals(otherEvent));
    /* Événements maintenant pareils */
    otherEvent = new Evenement("Randonnée", "Mont Blanc", 8, Month.JANUARY,
        2023, 8, 0, 280, 18);
    assertTrue(this.event.equals(otherEvent));
  }
}
