package association;

import java.time.LocalDateTime;
import java.time.Month;
import java.time.format.DateTimeFormatter;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

/**
 * Description d'un événement d'une association : nom, lieu, date, durée,
 * participants et nombre maximum de participants.
 *
 * @author Jarod Martin
 */
public class Evenement implements java.io.Serializable {
  
  /**
   * Version de la classe pour la sérialisation.
   */
  private static final long serialVersionUID = 4161175371517199405L;
  
  /**
   * Nom de l'événement.
   */
  private String nom;
  
  /**
   * Lieu de l'événement.
   */
  private String lieu;
  
  /**
   * Date de l'événement.
   */
  private LocalDateTime date;
  
  /**
   * Durée en minute de l'événement.
   */
  private int duree;
  
  /**
   * Nombre maximum de participants.
   */
  private int nbParticipantsMax;
  
  /**
   * Participants inscrits à l'événement.
   */
  private Set<InterMembre> participants;
  
  /**
   * Crée un événement avec les informations essentielles.
   *
   * @param nom le nom (non null)
   * @param lieu le lieu (non null)
   * @param date la date (avec LocalDateNow non null)
   * @param duree la durée en minute supérieure à 0
   * @param nbParticipantsMax le nombre maximum de participants (0 signifie un
   *        nombre quelconque)
   */
  public Evenement(String nom, String lieu, LocalDateTime date, int duree,
      int nbParticipantsMax) {
    this.nom = nom != null ? nom : "";
    this.lieu = lieu != null ? lieu : "";
    this.date = date != null ? date : LocalDateTime.now();
    this.duree = duree > 0 ? duree : 1;
    this.nbParticipantsMax = nbParticipantsMax >= 0 ? nbParticipantsMax : 0;
    this.participants = new HashSet<>();
  }
  
  /**
   * Crée un événement avec les informations essentielles sans passer par un
   * objet LocalDateTime pour la date.
   *
   * @param nom le nom (non null)
   * @param lieu le lieu (non null)
   * @param jour le jour dans le mois (nombre de 0 à 31)
   * @param mois le mois dans l'année
   * @param annee l'année
   * @param heure l'heure de la journée (nombre entre 0 et 23)
   * @param minutes les minutes de l'heure (nombre entre 0 et 59)
   * @param duree la durée (en minutes)
   * @param nbParticipantsMax le nombre maximum de participants (0 signifie un
   *        nombre quelconque)
   */
  public Evenement(String nom, String lieu, int jour, Month mois, int annee,
      int heure, int minutes, int duree, int nbParticipantsMax) {
    /* TODO: à améliorer pour la vérification de la date. */
    this(nom, lieu, LocalDateTime.of(annee, mois, jour, heure, minutes), duree,
        nbParticipantsMax);
  }
  
  /**
   * Vérifie que l'événement ne chevauche pas un autre événement par rapport au
   * lieu (indépendamment du temps).
   *
   * @param evt l'événement (non null) à comparer
   * @return si les deux événements se chevauchent renvoie <code>false</code>,
   *         sinon renvoie <code>true</code> s'ils ne se chevauchent pas
   */
  public boolean pasDeChevauchementLieu(Evenement evt) {
    return !this.lieu.equals(evt.lieu);
  }
  
  /**
   * Vérifie que l'événement ne chevauche pas un autre événement par rapport au
   * temps (indépendamment du lieu). Un chevauchement a lorsque les dates de
   * début sont comprises entre la date de début et de fin de l'événement à
   * comparer et inversement. Cas particulier, il n'y a pas chevauchement
   * lorsque une date de fin correspond à une date de début.
   *
   * @param evt l'événement (non null) à comparer
   * @return si les deux événements se chevauchent renvoie <code>false</code>,
   *         sinon renvoie <code>true</code> s'ils ne se chevauchent pas
   */
  public boolean pasDeChevauchementTemps(Evenement evt) {
    LocalDateTime dateFin = this.date.plusMinutes(this.duree);
    LocalDateTime dateFinOther = evt.date.plusMinutes(evt.duree);
    /*
     * Si this.date est compris entre evt.date et dateFinOther, ou si evt.date
     * est compris entre this.date et dateFin, il y a chevauchement.
     */
    return !((this.date.isAfter(evt.date) && this.date.isBefore(dateFinOther))
        || (evt.date.isAfter(this.date) && evt.date.isBefore(dateFin))
        || this.date.isEqual(evt.date));
  }
  
  /**
   * Renvoie le nom de l'événement.
   *
   * @return le nom de l'événement
   */
  public String getNom() {
    return this.nom;
  }
  
  /**
   * Modifie le nom de l'événement.
   *
   * @param nom le nouveau nom (non null)
   */
  public void setNom(String nom) {
    if (nom != null) {
      this.nom = nom;
    }
  }
  
  /**
   * Renvoie le lieu de l'événement.
   *
   * @return le lieu de l'événement
   */
  public String getLieu() {
    return this.lieu;
  }
  
  /**
   * Modifie le lieu de l'événement.
   *
   * @param lieu le nouveau lieu (non null)
   */
  public void setLieu(String lieu) {
    if (lieu != null) {
      this.lieu = lieu;
    }
  }
  
  /**
   * Renvoie la date de l'événement.
   *
   * @return la date de l'événement
   */
  public LocalDateTime getDate() {
    return this.date;
  }
  
  /**
   * Modifie la date de l'événement.
   *
   * @param date la nouvelle date (objet LocalDateTime non null)
   */
  public void setDate(LocalDateTime date) {
    if (date != null) {
      this.date = date;
    }
  }
  
  /**
   * Modifie la date de l'événement.
   *
   * @param jour le jour dans le mois (nombre de 0 à 31)
   * @param mois le mois dans l'année
   * @param annee l'année
   * @param heure l'heure de la journée (nombre entre 0 et 23)
   * @param minutes les minutes de l'heure (nombre entre 0 et 59)
   */
  public void setDate(int jour, Month mois, int annee, int heure, int minutes) {
    /* TODO: à améliorer pour la vérification de la date. */
    this.date = LocalDateTime.of(annee, mois, jour, heure, minutes);
  }
  
  /**
   * Renvoie la durée en minute de l'événement.
   *
   * @return la durée de l'événement
   */
  public int getDuree() {
    return this.duree;
  }
  
  /**
   * Modifie la durée de l'événement.
   *
   * @param duree la nouvelle durée en minute (supérieure à 0)
   */
  public void setDuree(int duree) {
    if (duree > 0) {
      this.duree = duree;
    }
  }
  
  /**
   * Renvoie le nombre maximum de participants.
   *
   * @return le nombre maximum de participants
   */
  public int getNbParticipantsMax() {
    return this.nbParticipantsMax;
  }
  
  /**
   * Modifie le nombre maximum de participants.
   *
   * @param nbParticipantsMax le nouveau nombre maximum de participants (0
   *        signifie un nombre quelconque)
   */
  public void setNbParticipantsMax(int nbParticipantsMax) {
    if (nbParticipantsMax >= 0) {
      this.nbParticipantsMax = nbParticipantsMax;
    }
  }
  
  /**
   * Renvoie la liste des participants à l'événement.
   *
   * @return la liste des participants à l'événement
   */
  public Set<InterMembre> getParticipants() {
    return this.participants;
  }
  
  /**
   * Ajouter un membre à l'événement.
   *
   * @param membre le membre à ajouter (non null)
   * @return <code>true</code> s'il n'y a pas eu de problème, <code>false</code>
   *         si l'évènement est en conflit de calendrier avec un évènement
   *         auquel est déjà inscrit le membre ou si le nombre de participants
   *         maximum est déjà atteint
   */
  public boolean ajouterParticipant(InterMembre membre) {
    /* Si le membre est déjà inscrit */
    if (this.participants.contains(membre)) {
      return false;
    }
    /* Si le nombre de participants maximum a été atteint */
    if (this.participants.size() == this.nbParticipantsMax
        && this.nbParticipantsMax != 0) {
      return false;
    }
    /* Si un événement est en conflit. */
    for (Evenement event : membre.ensembleEvenements()) {
      if (!this.pasDeChevauchementTemps(event)) {
        return false;
      }
    }
    /* Ajout du participant. */
    this.participants.add(membre);
    /* Ajout de l'événement au membre */
    membre.ensembleEvenements().add(this);
    return true;
  }
  
  /**
   * Supprimer un participant de l'événement.
   *
   * @param membre le membre à supprimer (non null)
   * @return si le membre était bien inscrit à l'évènement, renvoie
   *         <code>true</code> pour préciser que l'annulation est effective,
   *         sinon <code>false</code> si le membre n'était pas inscrit à
   *         l'évènement
   */
  public boolean supprimerParticipant(InterMembre membre) {
    /* Si le membre est déjà inscrit. */
    if (!this.participants.contains(membre)) {
      return false;
    }
    /* Suppression du participant. */
    this.participants.remove(membre);
    /* Suppression de l'événement au membre. */
    membre.ensembleEvenements().remove(this);
    return true;
  }
  
  @Override
  public int hashCode() {
    return Objects.hash(this.nom, this.lieu, this.date, this.duree,
        this.nbParticipantsMax, this.participants);
  }
  
  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null) {
      return false;
    }
    if (!(obj instanceof Evenement)) {
      return false;
    }
    Evenement other = (Evenement) obj;
    return this.nom.equals(other.nom) && this.lieu.equals(other.lieu)
        && this.date.equals(other.date) && this.duree == other.duree
        && this.nbParticipantsMax == other.nbParticipantsMax
        && this.participants.equals(other.participants);
  }
  
  @Override
  public String toString() {
    String res = "Event: " + this.nom + " à " + this.lieu;
    // Formatage de la date sous un certain format.
    res += " le "
        + this.date.format(DateTimeFormatter.ofPattern("dd/MM/yyyy HH:mm"));
    res += " pendant " + this.duree + " minutes.";
    res += " Participants (max: " + this.nbParticipantsMax + ") {";
    for (InterMembre participant : this.participants) {
      res += participant.getInformationPersonnelle().getNom() + " "
          + participant.getInformationPersonnelle().getPrenom() + ", ";
    }
    if (this.participants.size() > 0) {
      res = res.substring(0, res.length() - 2);
    }
    return res + "}";
  }
  
  // A compléter :
  //
  // - génération automatique des getters, setters, constructeurs,
  // des méthodes hashCode, toString et equals.
  // Modifiez manuellement le code généré au besoin. Rajoutez notamment
  // les méthodes de gestion des participants à l'événement.
  //
  // - Rajoutez un/des constructeurs permettant de construire plus facilement
  // un événement sans avoir besoin de passer un paramètre de type LocalDateTime
  //
  // - Ecrivez la JavaDoc complète de la classe
}
