package com.services.impl;

import com.dtos.LieuDto;
import com.entities.Lieu;
import com.repositories.LieuRepository;
import com.services.LieuService;
import org.springframework.stereotype.Service;

import javax.persistence.EntityNotFoundException;
import java.util.ArrayList;
import java.util.List;

@Service("lieuService")
public class LieuServiceImpl implements LieuService {

    private final LieuRepository lieuRepository;

    public LieuServiceImpl(LieuRepository lieuRepository){
        this.lieuRepository = lieuRepository;
    }

    /**
    * Sauvegarde le lieu dans la base de données mariadb
    * @param lieuDto Le lieu à sauvegardé
    * @return  LieuDto le lieu sauvegardé
    */
    @Override
    public LieuDto saveLieu(LieuDto lieuDto) {
        Lieu lieu = lieuDtoToEntity(lieuDto);
        lieu = lieuRepository.save(lieu);
        return lieuEntityToDto(lieu);
    }

    /**
    * Récupère un lieu en fonction de sont id 
    * @param lieuId id du lieu que l'on veut récupérer
    * @return LieuDto le lieu possèdant cette id
    */
    @Override
    public LieuDto getLieuById(Long lieuId) {
        Lieu lieu = lieuRepository.findById(lieuId).orElseThrow(() -> new EntityNotFoundException("Lieux not found"));
        return lieuEntityToDto(lieu);
    }

    /**
    * Mets un lieu en fonction de sont id
    * @param lieuId l'id du lieu que l'on souhaite modifier
    * @param lieuDto Les nouvelle données du lieu
    * @return LieuDto Le nouveau lieu modifié
    */
    @Override
    public LieuDto updateLieu(Long lieuId, LieuDto lieuDto) {
        if(! lieuRepository.existsById(lieuId)) throw new EntityNotFoundException("Lieux not found");
        Lieu lieu = lieuDtoToEntity(lieuDto);
        lieu.setId(lieuId);
        lieuRepository.save(lieu);
        return lieuEntityToDto(lieu);
    }

    /**
     * Supprime un lieu en fonction de sont id dans la base de données
     * @param lieuId Id du lieu à supprimer
     * @return boolean
     */
    @Override
    public boolean deleteLieu(Long lieuId) {
        lieuRepository.deleteById(lieuId);
        return true;
    }

    /**
     * Récupère tous les lieux existants dans la base de données 
     * @param List<LieuDto> la liste des lieux
     */
    @Override
    public List<LieuDto> getAllLieu() {
        List<LieuDto> lieuDtos = new ArrayList<>();
        List<Lieu> lieus = lieuRepository.findAll();
        lieus.forEach(lieu -> {
            lieuDtos.add(lieuEntityToDto(lieu));
        });
        return lieuDtos;
    }


    /**
     * Convertis un lieuDto en lieu
     */
    private LieuDto lieuEntityToDto(Lieu lieu){
        LieuDto lieuDto = new LieuDto();
        lieuDto.setId(lieu.getId());
        lieuDto.setName(lieu.getName());
        lieuDto.setAddress(lieu.getAddress());
        return lieuDto;
    }

    /**
     * Convertis un lieu en lieuDto
     */
    private Lieu lieuDtoToEntity(LieuDto lieuDto){
        Lieu lieu = new Lieu();
        lieu.setName(lieuDto.getName());
        lieu.setAddress(lieuDto.getAddress());
        lieu.setId(lieuDto.getId());
        return lieu;
    }
}
