package com.controllers;

import com.dtos.LieuDto;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.bind.annotation.CrossOrigin;

import javax.annotation.PostConstruct;
import java.util.List;

@RestController
@RequestMapping("/lieux")
@CrossOrigin(origins = "http://localhost:8000")
public class LieuController {

    @Value("${API.LIEU.HOSTNAME}")
    private String hostnameApiLieu;

    @Value("${API.LIEU.PORT}")
    private String portApiLieu;

    private String  urlLieu;
    private RestTemplate restTemplate ;

    public LieuController(){
        this.restTemplate = new RestTemplate();
    }

    @PostConstruct
    public void init() {
        this.urlLieu = "http://" + this.hostnameApiLieu + ":" + this.portApiLieu + "/lieux";
    }

    @GetMapping
    public List<LieuDto> getLieu() {
        ResponseEntity<List<LieuDto>> responseEntity = restTemplate.exchange(
                this.urlLieu ,
                HttpMethod.GET,
                null,
                new ParameterizedTypeReference<List<LieuDto>>() {});

        if (responseEntity.getStatusCode() != HttpStatus.OK) {
            throw new RuntimeException("Erreur lors de la récupération des lieux depuis l'API Lieu");
        }
        return responseEntity.getBody();
    }

    /**
     * Créer une nouveau lieu dans le system
     */
    @PostMapping
    public LieuDto saveLieu(@RequestBody LieuDto lieuDto) {
        ResponseEntity<LieuDto> responseEntity = restTemplate.postForEntity(
                this.urlLieu,
                lieuDto,
                LieuDto.class
        );

        if (responseEntity.getStatusCode() != HttpStatus.OK) {
            throw new RuntimeException("Erreur lors de la création d'un nouveau lieu via l'API Lieu");
        }
        return responseEntity.getBody();
    }

    /**
     * Récupère un lieu basé sur son ID
     */
    @GetMapping("/{lieuxId}")
    public LieuDto lieuDto(@PathVariable Long lieuxId){
        ResponseEntity<LieuDto> responseEntity = restTemplate.exchange(
                this.urlLieu +"/"+lieuxId,
                HttpMethod.GET,
                null,
                new ParameterizedTypeReference<LieuDto>() {});

        if (responseEntity.getStatusCode() != HttpStatus.OK) {
            throw new RuntimeException("Erreur lors de la récupération des lieux depuis l'API Lieu");
        }
        return responseEntity.getBody();
    }

    /**
     * Mets à jour un membre basé sur son ID
     */
    @PutMapping("/{lieuxId}")
    public LieuDto updateLieuDto(@PathVariable Long lieuxId, final @RequestBody LieuDto lieuDto){
        ResponseEntity<LieuDto> responseEntity = restTemplate.exchange(
                this.urlLieu +"/"+lieuxId,
                HttpMethod.PUT,
                new HttpEntity<>(lieuDto),
                LieuDto.class
        );

        if (responseEntity.getStatusCode() != HttpStatus.OK) {
            throw new RuntimeException("Erreur lors de la création d'un nouveau lieu via l'API Lieu");
        }
        return responseEntity.getBody();

    }

    /**
     * Supprime un membre par sont Id
     */
    @DeleteMapping("/{lieuxId}")
    public Boolean deleteLieux(@PathVariable Long lieuxId){
        ResponseEntity<Boolean> responseEntity = restTemplate.exchange(
                this.urlLieu+"/"+lieuxId,
                HttpMethod.DELETE,
                null,
                Boolean.class
        );

        if (responseEntity.getStatusCode() != HttpStatus.OK) {
            throw new RuntimeException("Erreur lors de la suppression du lieu avec l'ID : " + lieuxId);
        }

        return responseEntity.getBody();
    }
}
