package association;

import java.time.LocalDateTime;
import java.time.Month;
import java.util.ArrayList;
import java.util.List;

/**
 * Gestionnaire des événements de l'association (ajout, suppression, ...).
 *
 * @author Jarod Martin
 *
 */
public class GestionEvenements implements InterGestionEvenements {
  
  /**
   * Liste des événements.
   */
  private List<Evenement> evenements;
  
  /**
   * Crée et initialise le gestionnaire d'événements sans événement pas défaut.
   */
  public GestionEvenements() {
    this.evenements = new ArrayList<>();
  }
  
  /**
   * Crée et initialise le gestionnaire d'événements avec une collection
   * d'événements donnée.
   *
   * @param evenements la collection d'événements
   */
  public GestionEvenements(List<Evenement> evenements) {
    this.evenements = evenements;
  }
  
  @Override
  public Evenement creerEvenement(String nom, String lieu, int jour, Month mois,
      int annee, int heure, int minutes, int duree, int nbParticipants) {
    /* Si la date est invalide */
    if (jour < 1 || jour > 31 || mois == null || heure < 0 || heure > 23
        || minutes < 0 || minutes > 60) {
      return null;
    }
    Evenement event = new Evenement(nom, lieu, jour, mois, annee, heure,
        minutes, duree, nbParticipants);
    /* Si un événement est en conflit. */
    for (Evenement otherEvent : this.evenements) {
      if (!event.pasDeChevauchementTemps(otherEvent)
          && !event.pasDeChevauchementLieu(event)) {
        return null;
      }
    }
    /* Ajout de l'événement */
    this.evenements.add(event);
    return event;
  }
  
  @Override
  public void supprimerEvenement(Evenement evt) {
    /* Si l'événement n'est pas dans la liste des événements. */
    if (!this.evenements.contains(evt)) {
      return;
    }
    /* Désincription des membres à l'événement. */
    for (InterMembre membre : evt.getParticipants()) {
      evt.supprimerParticipant(membre);
    }
    /* Suppression de l'évenement. */
    this.evenements.remove(evt);
  }
  
  @Override
  public List<Evenement> ensembleEvenements() {
    return this.evenements;
  }
  
  @Override
  public List<Evenement> ensembleEvenementAvenir() {
    List<Evenement> eventsAvenir = new ArrayList<>();
    /* Date à cet instant */
    LocalDateTime dateNow = LocalDateTime.now();
    /* Parcours de la liste des evenements. */
    for (Evenement event : this.evenements) {
      /* Si il est à venir, on l'ajoute */
      if (event.getDate().isAfter(dateNow)) {
        eventsAvenir.add(event);
      }
    }
    return eventsAvenir;
  }
  
  @Override
  public boolean inscriptionEvenement(Evenement evt, InterMembre mbr) {
    if (evt == null || mbr == null) {
      return false;
    }
    return evt.ajouterParticipant(mbr);
  }
  
  @Override
  public boolean annulerEvenement(Evenement evt, InterMembre mbr) {
    if (evt == null || mbr == null) {
      return false;
    }
    return evt.supprimerParticipant(mbr);
  }
  
}
