package com.services.impl;

import com.dtos.MembreDto;

import com.entities.Membre;
import com.repositories.MembreRepository;
import com.services.MembreService;
import org.springframework.stereotype.Service;

import javax.persistence.EntityExistsException;
import javax.persistence.EntityNotFoundException;
import java.util.ArrayList;
import java.util.List;

@Service("membreService")
public class MembreServiceImpl implements MembreService {

    private final MembreRepository membreRepository;

    public MembreServiceImpl(MembreRepository membreRepository){
        this.membreRepository = membreRepository;
    }

    /**
     * Sauvegarde un membre dans la base de données s'il n'éxiste pas un autre membre avec le même nom et prénom.
     * @param membreDto le membre à sauvegardé
     * @return membreDto le membre sauvegardé
     */
    @Override
    public MembreDto saveMembre(MembreDto membreDto) {
        if(this.membreDtoAlreadyExistByName(membreDto)) throw new EntityExistsException("Membre éxite déjà");
        Membre membre = membreDtoToEntity(membreDto);
        membre = membreRepository.save(membre);
        return membreEntityToDto(membre);
    }

    /**
    * Récupère un membre en fonction de sont id
    * @return MembreDto le membre possèdant l'id 
    */
    @Override
    public MembreDto getMembreById(Long membreId) {
        Membre membre = membreRepository.findById(membreId).orElseThrow(() -> new EntityNotFoundException("Membre n'éxiste pas"));
        return membreEntityToDto(membre);
    }


    /**
    * Supprime un membre en fonction de sont id 
    * @return boolean
    */
    @Override
    public boolean deleteMembre(Long memberId) {
        membreRepository.deleteById(memberId);
        return true;
    }

    /**
    * Mets à jour un membre en fonction de sont id
    * @param membreId L'id du membre à modifié
    * @param membreDto Les informations du membre à modifié
    * @return MembreDto le membre mis à jour
    */
    @Override
    public MembreDto updateMembre(Long membreId, MembreDto membreDto) {
        if(! membreRepository.existsById(membreId)) throw new EntityNotFoundException("Membre n'éxiste pas");
        if(this.membreDtoAlreadyExistByName(membreDto)) throw new EntityExistsException("Membre éxite déjà");
        Membre membre = membreDtoToEntity(membreDto);
        membre.setId(membreId);
        membreRepository.save(membre);
        return membreEntityToDto(membre);

    }

    /**
     * Récupère tous les membres
     * @return List<MembreDto> 
     */
    @Override
    public List<MembreDto> getAllMembres() {
        List<MembreDto> membresDtos = new ArrayList<>();
        List<Membre> membres = membreRepository.findAll();
        membres.forEach( lieu -> {
            membresDtos.add(membreEntityToDto(lieu));
        });
        return membresDtos;
    }

    /**
     * Convertis un lieuDto en lieu
     * @return MembreDto
     */
    private MembreDto membreEntityToDto(Membre membre){
        MembreDto membreDto = new MembreDto();

        membreDto.setId(membre.getId());
        membreDto.setAdresse(membre.getAdresse());
        membreDto.setNom(membre.getNom());
        membreDto.setPrenom(membre.getPrenom());
        membreDto.setMail(membre.getMail());
        membreDto.setDateNaissance(membre.getDateNaissance());
        membreDto.setMotDePasse(membre.getMotDePasse());
        return membreDto;
    }

    /**
     * Convertis lieu  en lieuDto
     * @return Membre
     */
    private Membre membreDtoToEntity(MembreDto membreDto){
        Membre membre = new Membre();

        membre.setId(membreDto.getId());
        membre.setAdresse(membreDto.getAdresse());
        membre.setNom(membreDto.getNom());
        membre.setPrenom(membreDto.getPrenom());
        membre.setMail(membreDto.getMail());
        membre.setDateNaissance(membreDto.getDateNaissance());
        membre.setMotDePasse(membreDto.getMotDePasse());
        return membre;
    }

    /**
     * Indique si un membre exite déjà.
     * @param membreDto
     * @return Boolean true s'il existe déjà sinon false
     */
    private boolean membreDtoAlreadyExistByName(MembreDto membreDto){
        List<MembreDto> membres = this.getAllMembres();
        for(MembreDto membre : membres){
            if( membre.getNom().equals(membreDto.getNom()) && membre.getPrenom().equals(membreDto.getPrenom())) {
                return true;
            }
        }
        return false;
    }


}
